"""Slovní fotbal - rozhodne, zda lze ze slovníku sestavit řetěz používající všechna slova."""
import random
import sys
from typing import Optional

from slovni_pomucky import nacti_slovnik, posledni_pismeno, prvni_pismeno

# Umístění slovníku, pokud je program spuštěn bez argumentů
# Umístění slovníku lze programu dát jako první argument,
# např: python3 02_slovni_fotbal_retez.py input/jde01.txt
VYCHOZI_CESTA_K_SLOVNIKU = 'inputs_retez/jde04.txt'

Slovnik = dict[str, set[str]]

# Slovní fotbal = orientovaný graf, kde uzly jsou písmena a hrany jsou slova.
# Hledáme Eulerovskou cestu = cestu procházející každou hranou právě jednou.
# Podmínky existence: graf je souvislý a nejvýše jeden uzel má o 1 více výchozích
# hran (startovní písmeno) a nejvýše jeden o 1 více příchozích (koncové písmeno).


def urci_slovo(
        posledni: Optional[str],
        podle_zacatku: Slovnik,
        pouze_od_pismen: Optional[list[str]] = None,
) -> Optional[str]:
    """Vyber a odeber ze slovníku slovo navazující na posledni."""
    if posledni is None:
        if pouze_od_pismen is None:
            pocatecni_pismeno = random.choice(
                [k for k in podle_zacatku.keys() if len(podle_zacatku[k]) > 0]
            )
        else:
            pocatecni_pismeno = random.choice(pouze_od_pismen)
    else:
        pocatecni_pismeno = posledni_pismeno(posledni)
    if len(podle_zacatku.get(pocatecni_pismeno, [])) == 0:
        return None
    return podle_zacatku[pocatecni_pismeno].pop()


def vytvor_kruznici(
        podle_zacatku: Slovnik,
        pismena: Optional[list[str]] = None,
) -> list[str]:
    """
    Vytvoř kružnici v grafu: sled slov začínající a končící stejným písmenem.
    Slova jsou odebírána ze slovníku.
    """
    posledni: Optional[str] = None
    slova_v_kruznici: list[str] = []
    while True:
        posledni = urci_slovo(posledni, podle_zacatku, pismena)
        if posledni is None:
            break
        slova_v_kruznici.append(posledni)
    return slova_v_kruznici


def najdi_koncove_uzly(
        podle_zacatku: Slovnik,
        podle_konce: Slovnik,
) -> Optional[tuple[Optional[str], Optional[str]]]:
    """
    Zkontroluje, zda Eulerovská cesta může existovat, a vrátí (pocatecni, koncove).
    Pokud podmínky nejsou splněny, vrátí None.

    Pro každé písmeno musí platit:
      počet slov začínajících tímto písmenem == počet slov končících tímto písmenem.
    Výjimkou jsou nejvýše dvě písmena:
      startovní (o jedno více výchozích hran) a koncové (o jedno více příchozích).
    """
    vsechna_pismena: set[str] = set(podle_zacatku.keys()) | set(podle_konce.keys())
    pocty_zacatku: dict[str, int] = {
        p: len(podle_zacatku.get(p, [])) for p in vsechna_pismena
    }
    pocty_koncu: dict[str, int] = {
        p: len(podle_konce.get(p, [])) for p in vsechna_pismena
    }

    kandidati_startu: list[str] = [
        p for p in podle_zacatku if pocty_zacatku[p] == pocty_koncu[p] + 1
    ]
    kandidati_konce: list[str] = [
        p for p in podle_konce if pocty_zacatku[p] + 1 == pocty_koncu[p]
    ]
    problematicka_pismena: set[str] = (
            {p for p in vsechna_pismena if pocty_zacatku[p] != pocty_koncu[p]}
            - set(kandidati_startu) - set(kandidati_konce)
    )

    if problematicka_pismena or len(kandidati_startu) > 1 or len(kandidati_konce) > 1:
        return None

    pocatecni: Optional[str] = (
        podle_zacatku[kandidati_startu[0]].pop() if kandidati_startu else None
    )
    koncove: Optional[str] = None
    if kandidati_konce:
        koncove = podle_konce[kandidati_konce[0]].pop()
        podle_zacatku[prvni_pismeno(koncove)].remove(koncove)

    return pocatecni, koncove


def hierholzer(
        podle_zacatku: Slovnik,
        pocatecni: Optional[str],
        koncove: Optional[str],
) -> list[str]:
    """
    Hierholzerův algoritmus: sestaví Eulerovskou cestu z kružnic.
    Postupně vkládá nalezené kružnice do hlavní cesty na místě, kde navazují.
    """
    retez: list[str] = ([] if pocatecni is None else [pocatecni])
    retez += vytvor_kruznici(
        podle_zacatku, [posledni_pismeno(pocatecni)] if pocatecni else None
    )
    if koncove is not None:
        retez.append(koncove)

    while True:
        novy_retez: list[str] = []
        for slovo in retez:
            novy_retez.append(slovo)
            kruznice: list[str] = vytvor_kruznici(
                podle_zacatku, [posledni_pismeno(slovo)]
            )
            if kruznice:
                novy_retez += kruznice
        if len(retez) == len(novy_retez):
            break
        retez = novy_retez

    return retez


def main() -> None:
    """Načte slovník, spustí algoritmus a vypíše výsledek."""
    cesta_k_slovniku: str = sys.argv[1] if len(sys.argv) > 1 else VYCHOZI_CESTA_K_SLOVNIKU
    slova_podle_zacatku, slova_podle_konce = nacti_slovnik(cesta_k_slovniku)


    koncove_uzly = najdi_koncove_uzly(slova_podle_zacatku, slova_podle_konce)
    if koncove_uzly is None:
        print("NE")
        sys.exit(0)

    print(koncove_uzly)
    pocatecni_slovo, koncove_slovo = koncove_uzly
    retez: list[str] = hierholzer(slova_podle_zacatku, pocatecni_slovo, koncove_slovo)

    try:
        if all(not v for v in slova_podle_zacatku.values()):
            print("ANO")
            print(",".join(retez))
        else:
            print("NE")
    except BrokenPipeError:
        # Pro pouziti `python3 02_slovni_fotbal_retez.py | head -n 1`
        sys.exit(0)


main()
