import math
import time
import tkinter as tk
from dataclasses import dataclass
from datetime import datetime, timedelta, timezone
from tkinter import messagebox
from typing import Callable, Optional


@dataclass
class ClockConfig:
    name: str
    mode: str  # "local", "utc", "fixed"
    offset_hours: int = 0
    fixed_time: str = "00:00:00"


class TimeProvider:
    def __init__(self, config: ClockConfig) -> None:
        self.config = config
        self._fixed_base: Optional[datetime] = None
        self._fixed_start: Optional[float] = None
        if self.config.mode == "fixed":
            self._init_fixed_time()

    def _init_fixed_time(self) -> None:
        try:
            parts = [int(p) for p in self.config.fixed_time.split(":")]
            if len(parts) != 3:
                raise ValueError
            h, m, s = parts
            self._fixed_base = datetime.now().replace(hour=h, minute=m, second=s, microsecond=0)
            self._fixed_start = time.perf_counter()
        except ValueError:
            self._fixed_base = None
            self._fixed_start = None

    def now(self) -> datetime:
        if self.config.mode == "fixed":
            if self._fixed_base and self._fixed_start is not None:
                elapsed = time.perf_counter() - self._fixed_start
                return self._fixed_base + timedelta(seconds=elapsed)
            return datetime.now()
        if self.config.mode == "utc":
            base = datetime.now(timezone.utc).replace(tzinfo=None)
        else:
            base = datetime.now()
        return base + timedelta(hours=self.config.offset_hours)


class AnalogClockView(tk.Frame):
    def __init__(self, master: tk.Widget, config: ClockConfig) -> None:
        super().__init__(master, bd=1, relief="groove")
        self.config_data = config
        self.provider = TimeProvider(config)
        self.size = 180
        self.radius = 75
        self.center = (self.size // 2, self.size // 2)
        self.canvas = tk.Canvas(self, width=self.size, height=self.size, bg="white")
        self.canvas.pack(padx=6, pady=6)
        self.label = tk.Label(self, text=config.name or "Hodiny")
        self.label.pack(pady=(0, 6))
        self._draw_face()

    def _draw_face(self) -> None:
        cx, cy = self.center
        r = self.radius
        self.canvas.create_oval(
            cx - r,
            cy - r,
            cx + r,
            cy + r,
            width=2,
            outline="black",
            tags="face",
        )
        for i in range(12):
            angle = math.radians(i * 30 - 90)
            x1 = cx + math.cos(angle) * (r - 8)
            y1 = cy + math.sin(angle) * (r - 8)
            x2 = cx + math.cos(angle) * (r - 2)
            y2 = cy + math.sin(angle) * (r - 2)
            self.canvas.create_line(x1, y1, x2, y2, width=2, fill="black", tags="face")
            num_radius = r - 20
            nx = cx + math.cos(angle) * num_radius
            ny = cy + math.sin(angle) * num_radius
            number = 12 if i == 0 else i
            self.canvas.create_text(
                nx,
                ny,
                text=str(number),
                font=("Helvetica", 10, "bold"),
                fill="black",
                tags="face",
            )

    def update_time(self) -> None:
        now = self.provider.now()
        self.canvas.delete("hands")
        cx, cy = self.center
        r = self.radius
        sec = now.second + now.microsecond / 1_000_000
        minute = now.minute + sec / 60
        hour = (now.hour % 12) + minute / 60
        # Uhel rucicek: 0 stupnu je 12 hodin (nahoru), proto -90 stupnu.
        sec_angle = math.radians(sec * 6 - 90)
        min_angle = math.radians(minute * 6 - 90)
        hour_angle = math.radians(hour * 30 - 90)
        self._hand(cx, cy, sec_angle, r * 0.9, "red", 1)
        self._hand(cx, cy, min_angle, r * 0.75, "black", 2)
        self._hand(cx, cy, hour_angle, r * 0.55, "black", 3)

    def _hand(self, cx: int, cy: int, angle: float, length: float, color: str, width: int) -> None:
        x = cx + math.cos(angle) * length
        y = cy + math.sin(angle) * length
        self.canvas.create_line(cx, cy, x, y, fill=color, width=width, tags="hands")

    def update_config(self, config: ClockConfig) -> None:
        self.config_data = config
        self.provider = TimeProvider(config)
        self.label.config(text=config.name or "Hodiny")


class ClockManager(tk.Frame):
    def __init__(self, master: tk.Widget) -> None:
        super().__init__(master)
        header = tk.Frame(self)
        header.pack(fill="x")
        tk.Label(header, text="Hodiny", font=("Helvetica", 12, "bold")).pack(side="left")
        tk.Button(header, text="Přidat hodiny", command=self._open_add_dialog).pack(side="right")
        self.container = tk.Frame(self)
        self.container.pack(fill="both", expand=True)
        self.clocks: list[AnalogClockView] = []
        self._relayout()

    def _relayout(self) -> None:
        for i, clock in enumerate(self.clocks):
            clock.grid_forget()
            row = i // 3
            col = i % 3
            clock.grid(row=row, column=col, padx=6, pady=6)

    def add_clock(self, config: ClockConfig) -> None:
        view = AnalogClockView(self.container, config)
        menu = tk.Frame(view)
        menu.pack(pady=(0, 4))
        tk.Button(menu, text="Upravit", command=lambda v=view: self._open_edit_dialog(v)).pack(side="left")
        tk.Button(menu, text="Odebrat", command=lambda v=view: self.remove_clock(v)).pack(side="left", padx=(6, 0))
        self.clocks.append(view)
        self._relayout()

    def remove_clock(self, view: AnalogClockView) -> None:
        self.clocks.remove(view)
        view.destroy()
        self._relayout()

    def update_all(self) -> None:
        for clock in self.clocks:
            clock.update_time()

    def _open_add_dialog(self) -> None:
        ClockDialog(self, on_submit=self.add_clock)

    def _open_edit_dialog(self, view: AnalogClockView) -> None:
        def on_submit(config: ClockConfig) -> None:
            view.update_config(config)
        ClockDialog(self, initial=view.config_data, on_submit=on_submit)


class ClockDialog(tk.Toplevel):
    def __init__(
        self,
        master: tk.Widget,
        initial: Optional[ClockConfig] = None,
        on_submit: Optional[Callable[[ClockConfig], None]] = None,
    ) -> None:
        super().__init__(master)
        self.title("Nastavení hodin")
        self.resizable(False, False)
        self.on_submit = on_submit
        self.transient(master)
        self.grab_set()

        self.name_var = tk.StringVar(value=initial.name if initial else "")
        self.mode_var = tk.StringVar(value=initial.mode if initial else "local")
        self.offset_var = tk.StringVar(value=str(initial.offset_hours) if initial else "0")
        self.fixed_var = tk.StringVar(value=initial.fixed_time if initial else "00:00:00")

        tk.Label(self, text="Název:").grid(row=0, column=0, sticky="w", padx=6, pady=4)
        tk.Entry(self, textvariable=self.name_var, width=22).grid(row=0, column=1, padx=6, pady=4)

        tk.Label(self, text="Režim:").grid(row=1, column=0, sticky="w", padx=6, pady=4)
        mode_frame = tk.Frame(self)
        mode_frame.grid(row=1, column=1, padx=6, pady=4, sticky="w")
        for text, value in (
            ("Lokální + posun", "local"),
            ("UTC + posun", "utc"),
            ("Zadaný čas", "fixed"),
        ):
            tk.Radiobutton(mode_frame, text=text, value=value, variable=self.mode_var).pack(anchor="w")

        tk.Label(self, text="Posun (hod, +/-):").grid(row=2, column=0, sticky="w", padx=6, pady=4)
        tk.Entry(self, textvariable=self.offset_var, width=10).grid(row=2, column=1, padx=6, pady=4, sticky="w")

        tk.Label(self, text="Zadaný čas (HH:MM:SS):").grid(row=3, column=0, sticky="w", padx=6, pady=4)
        tk.Entry(self, textvariable=self.fixed_var, width=12).grid(row=3, column=1, padx=6, pady=4, sticky="w")

        button_frame = tk.Frame(self)
        button_frame.grid(row=4, column=0, columnspan=2, pady=8)
        tk.Button(button_frame, text="OK", command=self._submit).pack(side="left", padx=6)
        tk.Button(button_frame, text="Zrušit", command=self.destroy).pack(side="left")

    def _submit(self) -> None:
        name = self.name_var.get().strip()
        if not name:
            messagebox.showerror("Chyba", "Zadejte název hodin.")
            return
        try:
            offset = int(self.offset_var.get().strip())
        except ValueError:
            messagebox.showerror("Chyba", "Posun musí být celé číslo (hodiny).")
            return
        fixed = self.fixed_var.get().strip()
        if self.mode_var.get() == "fixed":
            if not self._validate_fixed_time(fixed):
                messagebox.showerror("Chyba", "Fixní čas musí být ve formátu HH:MM:SS.")
                return
        config = ClockConfig(name=name, mode=self.mode_var.get(), offset_hours=offset, fixed_time=fixed)
        if self.on_submit:
            self.on_submit(config)
        self.destroy()

    @staticmethod
    def _validate_fixed_time(value: str) -> bool:
        parts = value.split(":")
        if len(parts) != 3:
            return False
        try:
            h, m, s = (int(p) for p in parts)
        except ValueError:
            return False
        return 0 <= h < 24 and 0 <= m < 60 and 0 <= s < 60


class WorldClockApp(tk.Tk):
    def __init__(self) -> None:
        super().__init__()
        self.title("Světový čas")
        self.geometry("800x600")

        self.clock_manager = ClockManager(self)
        self.clock_manager.pack(fill="both", expand=True)

        self.clock_manager.add_clock(ClockConfig(name="Praha", mode="local"))
        self.clock_manager.add_clock(ClockConfig(name="UTC", mode="utc"))

        self._tick()

    def _tick(self) -> None:
        self.clock_manager.update_all()
        self.after(50, self._tick)


def main() -> None:
    app = WorldClockApp()
    app.mainloop()


if __name__ == "__main__":
    main()
