use clap::Parser;
use itertools::Itertools;
use log::info;
use rand::prelude::*;
use std::collections::{HashSet, VecDeque};
use std::fs::File;
use std::io::{BufWriter, Write};
use std::path::{Path, PathBuf};
use std::rc::Rc;
use std::{fs, io};

#[derive(Parser)]
struct Args {
    #[arg(long, short = 'f', default_value_t = 50)]
    file_size_min: usize,
    #[arg(long, short = 'F', default_value_t = 500)]
    file_size_max: usize,
    #[arg(long, short = 'd', default_value_t = 2)]
    depth_min: usize,
    #[arg(long, short = 'D', default_value_t = 10)]
    depth_max: usize,
    #[arg(long, short = 'l', default_value_t = 5)]
    link_count_mid: usize,
    #[arg(long, short = 'r', default_value_t = 0.5)]
    reuse_rate: f64,
    #[arg(long, short = 'L')]
    loop_rate: Option<f64>,
    #[arg(long, short = 'S', default_value_t = 0.5)]
    link_count_spread: f64,
    #[arg(long, short = 's')]
    rng_seed: Option<u64>,
    #[arg(long, short)]
    target_folder: PathBuf,
}

const FILENAME_CHARSET: [char; 26] = [
    'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', 'A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'Z', 'X', 'C', 'V',
    'B', 'N', 'M',
];

const CONTENT_CHARSET: [char; 54] = [
    'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', 'A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'Z', 'X', 'C', 'V',
    'B', 'N', 'M', ' ', 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', 'a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l',
    'z', 'x', 'c', 'v', 'b', 'n', 'm', '\n',
];

fn random_filename<T>(rng: &mut T) -> Rc<Path>
where
    T: Rng,
{
    PathBuf::from(
        (0..5)
            .map(|_| FILENAME_CHARSET.choose(rng).unwrap())
            .collect::<String>()
            + ".txt",
    )
    .into()
}

fn random_existing_link<T>(rng: &mut T, filenames: &HashSet<Rc<Path>>) -> Rc<Path>
where
    T: Rng,
{
    let filename = filenames
        .iter()
        .choose(rng)
        .expect("Infallible. Emptiness of set checked in advance.")
        .clone();
    info!(
        "Reusing filename: {}",
        filename.as_os_str().to_str().expect("Non unicode path is impossible.")
    );
    filename
}

fn random_new_link<T>(
    rng: &mut T,
    unprocessed_filenames: &mut HashSet<Rc<Path>>,
    work_queue: &mut VecDeque<(usize, Rc<Path>)>,
    depth: usize,
) -> Rc<Path>
where
    T: Rng,
{
    let new_filename = random_filename(rng);
    unprocessed_filenames.insert(new_filename.clone());
    work_queue.push_back((depth + 1, new_filename.clone()));
    info!(
        "New filename: {}",
        new_filename
            .as_os_str()
            .to_str()
            .expect("Non unicode path is impossible.")
    );
    new_filename
}

fn random_link<T>(
    rng: &mut T,
    processed_filenames: &HashSet<Rc<Path>>,
    unprocessed_filenames: &mut HashSet<Rc<Path>>,
    work_queue: &mut VecDeque<(usize, Rc<Path>)>,
    reuse_ratio: f64,
    loop_ratio: f64,
    depth: usize,
) -> Rc<Path>
where
    T: Rng,
{
    if loop_ratio > 0.0 {
        if !processed_filenames.is_empty() && rng.random_bool(loop_ratio) {
            info!("Making a loop");
            random_existing_link(rng, processed_filenames)
        } else if !unprocessed_filenames.is_empty() && rng.random_bool(reuse_ratio) {
            random_existing_link(rng, unprocessed_filenames)
        } else {
            random_new_link(rng, unprocessed_filenames, work_queue, depth)
        }
    } else {
        if !unprocessed_filenames.is_empty() && rng.random_bool(reuse_ratio) {
            random_existing_link(rng, unprocessed_filenames)
        } else {
            random_new_link(rng, unprocessed_filenames, work_queue, depth)
        }
    }
}

fn generate<T>(args: Args, rng: &mut T) -> io::Result<Rc<Path>>
where
    T: Rng,
{
    fs::create_dir_all(&args.target_folder)?;
    std::env::set_current_dir(&args.target_folder)?;
    let entrypoint = random_filename(rng);
    let depth_range = args.depth_max - args.depth_min;
    let mut work_queue: VecDeque<(usize, Rc<Path>)> = Default::default();
    let mut processed_filenames: HashSet<Rc<Path>> = Default::default();
    let mut unprocessed_filenames: HashSet<Rc<Path>> = Default::default();
    unprocessed_filenames.insert(entrypoint.clone());
    work_queue.push_back((0, entrypoint.clone()));
    while let Some((depth, filename)) = work_queue.pop_front() {
        if processed_filenames.insert(filename.clone()) {
            unprocessed_filenames.remove(&filename);
            let content_len = rng.random_range(args.file_size_min..args.file_size_max);
            let content = (0..content_len)
                .map(|_| CONTENT_CHARSET.choose(rng).unwrap())
                .collect::<String>();
            let depth_sat = depth.saturating_sub(args.depth_min);
            let depth_relative = depth_range - depth_sat;
            let link_max = args.link_count_mid * depth_relative / depth_range;
            let link_count = rng.random_range(
                (link_max as f64 * (1.0 - args.link_count_spread)) as usize
                    ..=(link_max as f64 * (1.0 + args.link_count_spread)) as usize,
            );
            let link_positions = (0..link_count)
                .map(|_| rng.random_range(0..=content_len))
                .sorted()
                .scan(0, |prev, curr| {
                    let next = curr - *prev;
                    *prev = curr;
                    Some(next)
                })
                .collect_vec();
            let mut file = BufWriter::new(File::create(&filename)?);
            let mut s: &str = content.as_str();
            info!(
                "File {} - length {content_len} - links {link_positions:?}",
                filename.as_os_str().to_str().expect("Non unicode path is impossible.")
            );
            for idx in link_positions {
                let (lhs, rhs) = s.split_at(idx);
                file.write(lhs.as_bytes())?;
                let linked_file = random_link(
                    rng,
                    &mut processed_filenames,
                    &mut unprocessed_filenames,
                    &mut work_queue,
                    (1.0 - depth_relative as f64 / depth_range as f64) * args.reuse_rate,
                    (1.0 - depth_relative as f64 / depth_range as f64) * args.loop_rate.unwrap_or_default(),
                    depth,
                );
                write!(
                    file,
                    "[{}]",
                    linked_file
                        .as_os_str()
                        .to_str()
                        .expect("Non unicode path is impossible.")
                )?;
                s = rhs;
            }
        }
    }

    Ok(entrypoint)
}

fn main() {
    env_logger::init();
    let args = Args::parse();
    let mut rng = StdRng::seed_from_u64(args.rng_seed.unwrap_or_else(|| rand::rng().random()));
    match generate(args, &mut rng) {
        Ok(entrypoint) => println!("Entrypoint is {}", entrypoint.as_os_str().to_str().unwrap()),
        Err(err) => {
            eprintln!("Failed to generate input: {err}")
        }
    };
}
