use quick_cache::sync::Cache;
use std::borrow::Cow;
use std::fmt::Debug;
use std::fs;
use std::io::Write;
use std::path::{Path, PathBuf};
use std::rc::Rc;

#[derive(Debug)]
pub enum Error {
    IO(std::io::Error),
    Logic(Vec<Rc<Path>>),
}

impl From<std::io::Error> for Error {
    fn from(value: std::io::Error) -> Self {
        Error::IO(value)
    }
}

impl From<Vec<Rc<Path>>> for Error {
    fn from(value: Vec<Rc<Path>>) -> Self {
        Error::Logic(value)
    }
}

fn defrag_inner<T>(
    source: Rc<Path>,
    output: &mut T,
    cache: &Cache<Rc<Path>, Cow<str>>,
    stack: &mut Vec<Rc<Path>>,
) -> Result<(), Error>
where
    T: Write,
{
    if stack.iter().filter(|x| **x == source).count() > 1 {
        Err(stack.clone().into())
    } else {
        let input = cache.get_or_insert_with(&source, || fs::read_to_string(source.as_ref()).map(|x| x.into()))?;
        input
            .split(|c: char| c == '[' || c == ']')
            .zip([false, true].into_iter().cycle())
            .try_for_each(|(content, is_link)| -> Result<(), Error> {
                if is_link {
                    let path: Rc<Path> = source
                        .parent()
                        .unwrap_or(PathBuf::default().as_path())
                        .join(content)
                        .canonicalize()?
                        .into();
                    stack.push(path.clone());
                    defrag_inner(path, output, cache, stack)?;
                    stack.pop();
                } else {
                    output.write(content.as_bytes())?;
                }
                Ok(())
            })
    }
}

pub fn defrag<T>(source: &Path, output: &mut T) -> Result<(), Error>
where
    T: Write,
{
    let path: Rc<Path> = source.into();
    defrag_inner(path.clone(), output, &Cache::new(100), &mut vec![path])
}

fn defrag_simple_inner<T>(source: Rc<Path>, output: &mut T, stack: &mut Vec<Rc<Path>>) -> Result<(), Error>
where
    T: Write,
{
    if stack.iter().filter(|x| **x == source).count() > 1 {
        Err(stack.clone().into())
    } else {
        let input = fs::read_to_string(&source).unwrap();
        input
            .split(|c: char| c == '[' || c == ']')
            .zip([false, true].into_iter().cycle())
            .try_for_each(|(content, is_link)| -> Result<(), Error> {
                if is_link {
                    let path: Rc<Path> = source
                        .parent()
                        .unwrap_or(PathBuf::default().as_path())
                        .join(content)
                        .canonicalize()?
                        .into();
                    stack.push(path.clone());
                    defrag_simple_inner(path, output, stack)?;
                    stack.pop();
                } else {
                    output.write(content.as_bytes())?;
                }
                Ok(())
            })
    }
}

pub fn defrag_simple<T>(source: &Path, output: &mut T) -> Result<(), Error>
where
    T: Write,
{
    let path: Rc<Path> = source.into();
    defrag_simple_inner(path.clone(), output, &mut vec![path])
}
